
; Steam Train Whistle. Requires white noise source using PIC12F617

	ERRORLEVEL -302
	ERRORLEVEL -306


	    list      p=12F617            ; list directive to define processor
     #include <p12F617.inc>        ; processor specific variable definitions


 __CONFIG   _CP_OFF & _BOR_ON & _MCLRE_OFF & _WDT_OFF & _PWRTE_ON &_INTRC_OSC_NOCLKOUT & _IOSCFS_8MHZ &_WRT_OFF

; Bank 0 RAM

FREQ1			equ	H'20'	; Frequency 1 for whistle 1
FREQ2			equ	H'21'	; Frequency 2 for whistle 2
FREQ3			equ	H'22'	; Frequency 3 for whistle 3
COUNT1		equ	H'23'	; counter for frequency 1
COUNT2		equ	H'24'	; counter for frequency 2
COUNT3		equ	H'25'	; counter for frequency 3
START			equ	H'26'	; whistle startup rate
STRT_WKG		equ	H'27'	; whistle startup rate working register
DOPPLER		equ	H'28'	; Doppler flag for frequency shift
ON_OFF		equ	H'29'	; whistle on or off
JUMPER		equ	H'2A'	; JP1, 2 & 3 flags
LENGTH			equ	H'2B'	; whistle length
LNGTH_WKG	equ	H'2C'	; whistle length working register	
EXTENSION		equ	H'2D'	; length extension
SXTN			equ	H'2E'	; divided by 16 value for whistle length in Doppler Effect
SHIFT1			equ	H'2F'	; first shift point in Doppler Effect
SHIFT2			equ	H'30'	; second shift point in Doppler Effect
PROCESSED	equ	H'31'	; processed value for frequency settings
OUTPUT			equ	H'32'	; GPIO value
CHANGED		equ	H'33'	; output changed flag
SPEED			equ	H'34'	; train speed for Doppler Effect
SW_READ		equ	H'35'	; switch setting
SHIFT_RAMP	equ	H'36'	; ramping up/down of frequency flag for when in Doppler
APROCH_AWAY 	equ	H'37'	; flag for direction of volume required 
MIDWAY			equ	H'38'	; midway for whistle used in Doppler Effect
VOL_RAMP		equ	H'39'	; position in LNGTH_WKG where volume is to reduce
DIV1_2			equ	H'3A'	; division by 8 for rectangular generation output1
DIV2_2			equ	H'3B'	; division by 8 for rectangular generation output2
DIV3_2			equ	H'3C'	; division by 8 for rectangular generation output3
VOLUME		equ	H'3D'	; volume
TEMP1			equ	H'3E'	; temporary

; All Banks RAM
W_TMP			equ	H'70'	; storage of w before interrupt
STATUS_TMP	equ	H'71'	; status storage before interrupt
STORE1			equ	H'72'	; delay store
STORE2			equ	H'73'	; delay store
TEMP			equ	H'74'	; temporary use
RAMP_END		equ	H'75'	; end point for frequency change in OSCTUNE

; DATA (4 blocks)
DATA1ms		equ	H'78'	; data of first byte ms
DATA1ls			equ	H'79'	; data
DATA2ms		equ	H'7A'	; data 
DATA2ls			equ	H'7B'	; data
DATA3ms		equ	H'7C'	; data 
DATA3ls			equ	H'7D'	; data
DATA4ms		equ	H'7E'	; data 
DATA4ls			equ	H'7F'	; data
;___________________________________________________________


	org	H'400'		; start address of where data memory is stored
; frequency set by counter that ranges from 20 to 80. 
; 20 is 1kHz, 80 is 250Hz
; f =1/ (2 x counter x 25us)

	DE 	D'27' 	; DATA1ls at 740Hz (adjustable with JP2 and JP3 in at power up using VR1)	
	DE 	D'38'	; DATA2ls at 526Hz for 525Hz (adjustable with JP1 and JP3 in at power up using VR1)
	DE 	D'48'	; DATA3ls at 416Hz for 420Hz (adjustable with JP1 and JP2 in at power up using VR1)
	DE 	D'03'	; whistle volume startup rate is adjusted with VR1 when JP1,2 & 3 all in at power up.
; For Doppler Effect it includes end rate simulating volume with distance
; anticlockwise to mid adjsutment is for non-Doppler and mid to fully clockwise is for Doppler
;___________________________________________________________
 
	org	0
	goto	SETUP
	org	4
	
; start interrupt by saving w and status registers  
INTERRUPT
	movwf	W_TMP			; w to w_tmp storage
	swapf	STATUS,w		; status to w
	movwf	STATUS_TMP	; status in status_tmp 

	bcf		STATUS,RP0	; select bank 0
	bcf		STATUS,RP1	; bank 0

	bcf		INTCON,T0IF		; clear timer0 overflow flag
	
	movf	GPIO,w
	movwf	SW_READ

; Add second order harmonics by using rectangular wave rather than square
; set interrupt rate
	movlw	D'208'			; leave 50 for counter for 50us interrupt rate
	addwf	TMR0,f	

; start with clearing flag
	clrf		CHANGED	 

; set frequency timers
ONE_SEC
	decfsz	COUNT1,f
	goto	ONE_SEC_TEST
; alternate GP1
	movf	FREQ1,w		; frequency value 
	movwf	COUNT1
	bsf		CHANGED,0		; set flag when output changed
	bcf		OUTPUT,1
	goto	TWO_SEC
	
ONE_SEC_TEST
	movf	COUNT1,w
	subwf	DIV1_2,w
	btfsc	STATUS,C
	goto	ONE_SEC_TEST2
; check output and change
	bsf		CHANGED,0		; set flag when output changed
	bcf		OUTPUT,1
	goto	TWO_SEC

ONE_SEC_TEST2
	bsf		CHANGED,0
	bsf		OUTPUT,1

TWO_SEC
; set frequency timers
	decfsz	COUNT2,f
	goto	TWO_SEC_TEST
; alternate GP1
	movf	FREQ2,w		; frequency value 
	movwf	COUNT2
	bsf		CHANGED,0		; set flag when output changed
	bcf		OUTPUT,0
	goto	THREE_SEC
	
TWO_SEC_TEST
	movf	COUNT2,w
	subwf	DIV2_2,w
	btfsc	STATUS,C
	goto	TWO_SEC_TEST2
; check output and change
	bsf		CHANGED,0		; set flag when output changed
	bcf		OUTPUT,0
	goto	THREE_SEC

TWO_SEC_TEST2
	bsf		CHANGED,0		; set flag when output changed
	bsf		OUTPUT,0

THREE_SEC
; set frequency timers
	decfsz	COUNT3,f
	goto	THREE_SEC_TEST
; alternate GP1
	movf	FREQ3,w		; frequency value 
	movwf	COUNT3
	bsf		CHANGED,0		; set flag when output changed
	bcf		OUTPUT,5
	goto	SET_OUT
	
THREE_SEC_TEST
	movf	COUNT3,w
	subwf	DIV3_2,w
	btfsc	STATUS,C
	goto	THREE_SEC_TEST2
; check output and change
	bsf		CHANGED,0		; set flag when output changed
	bcf		OUTPUT,5
	goto	SET_OUT

THREE_SEC_TEST2
	bsf		CHANGED,0		; set flag when output changed
	bsf		OUTPUT,5

SET_OUT
	btfss	CHANGED,0		; check if port change needed
	goto	RECLAIM
	movf	OUTPUT,w		; output required
; move output values to GPIO when outputs are changed (changed,0 is set)
	movwf	GPIO

; end of interrupt 
RECLAIM

; reclaim w and status
	swapf	STATUS_TMP,w	; status temp storage to w
	movwf	STATUS			; w to status register
	swapf	W_TMP,f		; swap upper and lower 4-bits in w_tmp
	swapf   	W_TMP,w		; swap bits and into w register
	retfie					; return from interrupt

; ******************************************************************
SETUP
; select speed for Doppler Effect	 (remark in or out)
;	movlw	D'15'			; 135km/h
;	movlw	D'12'			; 100km/h
	movlw	D'10'			; 80km/h	
	movwf	SPEED			; used in Doppler simulation

; set oscillator calibration, 
	bsf		STATUS,RP0	; bank 1
        movlw   D'00'			; set oscillator to factory calibrated frequency 
	movwf	OSCTUNE		; normal frequency
	bcf		STATUS,RP0	; bank 0
  
; set inputs/outputs
	movlw	B'00000000'
	movwf	GPIO			; ports low
	movlw	B'00000111'		; comparators off
	movwf	CMCON0
	bsf		STATUS,RP0	; select memory bank 1
	movlw	B'00101011'		; pullups on.
	movwf	WPU
	movlw	B'00111011'		; outputs/inputs set 
	movwf	TRISIO			; port data direction register
	movlw	B'00000000'		; settings 
	movwf	OPTION_REG

; analog inputs, A/D
	movlw	B'01011000'
	movwf	ANSEL			; digital I/O and analog at AN3
	bcf		STATUS,RP0	; select memory bank 0
	movlw	B'00001100'		; channel 3, left justified, VDD ref etc
	movwf	ADCON0
	bsf		ADCON0,0		; A/D on

; pwm set
	bsf		STATUS,RP0	; select memory bank 1
	movlw	D'101'			; H'65'
	movwf	PR2				; PWM period register for 19.61kHz 8-bit resolution if ls bits included in CCP1CON bits 5 and 4
	bcf		STATUS,RP0	; memory bank 0
	movlw	D'0'				; zero initially
	movwf	CCPR1L		; ms byte of PWM
	movlw	B'00000000'		; prescaler /1 for 7.8125kHz
	movwf	T2CON
	bsf		T2CON,2		; enable timer 2
	movlw	B'00001100'		; set PWM mode
	movwf	CCP1CON		; enable PWM operation

; power up delay
	movlw	D'255'
	call		DELAY2			; ~200ms
	movlw	D'255'
	call		DELAY2			; ~200ms

; read data
	movlw	D'0'				; DATA1
	movwf	TEMP			; ls byte of data address
	call		READ			; ls data in w
	movwf	FREQ1			; frequency 1
	movlw	D'1'				; DATA2
	movwf	TEMP			; ls byte of data address
	call		READ			; data in w
	movwf	FREQ2			; frequency 2
	movlw	D'2'				; DATA3
	movwf	TEMP			; ls byte of data address
	call		READ			; data in w
	movwf	FREQ3			; frequency 3
	movlw	D'3'				; DATA3
	movwf	TEMP			; ls byte of data address
	call		READ			; data in w
	movwf	START			; whistle startup volume rate
	movf	TEMP,w			; ms byte
	movwf	DOPPLER		; if bit 0 is set then Doppler mode

; initial
	clrf		COUNT1		; counters at 0
	clrf		COUNT2
	clrf		COUNT3
	clrf		ON_OFF		; whistle on or off
	clrf		EXTENSION		; length extension
	bsf		INTCON,T0IE		; enable timer 0 interrrupt

POWER_UP
; POWER UP Checks for JP1, JP2 and JP3
; GP0, GP1 and GP5 are initially inputs with pullups
; so reads high if a jumper out and a low if a jumper is in

; read GPIO. Pins that are loaded with a shunt in JP1, JP2 or JP3 will read as a low
	movf	GPIO,w
	movwf	TEMP

; set GPIO,0,1,5 as outputs 
	bsf		STATUS,RP0	; select memory bank 1
	movlw	B'00011000'		; outputs/inputs set 
	movwf	TRISIO			; port data direction register
	bcf		STATUS,RP0	; select memory bank 0

; set JUMPER bits at bits 1,2,3 for GPIO,1, GPIO,0 and GPIO,5 if read low
	clrf		JUMPER		; initially, jumper flags off
	btfss	TEMP,0			; bit 0 (GPIO,0)  if clear, set JUMPER,2 for JP2
	bsf		JUMPER,2
	btfss	TEMP,1			; bit1 (GPIO,1)  if clear, set JUMPER,1 for JP1
	bsf		JUMPER,1
	btfss	TEMP,5			; bit 5 (GPIO,5) if clear, set JUMPER,3 for JP3
	bsf		JUMPER,3
; check JUMPER
	movf	JUMPER,w
	btfsc	STATUS,Z		; if zero then normal run. If not zero it's setup mode
	goto	LOOP_START	; no jumpers so bypass setup

;________________________________________________________
; setup

; set volume 
	movlw	D'20'			; volume setting
	movwf	CCPR1L		; ms byte of PWM
	
; start  Interrupt for output frequencies generation
; run interrupt
	bcf		INTCON,T0IF		;  flag	 
	bsf 		INTCON,GIE		; set global interrupt enable 

; read VR1 and place in FREQ1,2 or 3 or START

SET_LOOP
	call		ACQUIRE_AD	; get VR1 value 
	comf	ADRESH,w		; reverse VR1 for frequency
	movwf	PROCESSED

; for frequency keep between 20 and 80 (say 63 range from 19 to 82) 19=1.053kHz, 82=244Hz
; Process for frequency
; divide by 4
	bcf		STATUS,C
	rrf		PROCESSED,f	
	bcf		STATUS,C
	rrf		PROCESSED,w
; add 19
	addlw	D'19'
	movwf	PROCESSED

	movlw	D'255'
	call		DELAY2			; ~200ms

; store data when S1 pressed
	btfss	GPIO,3
	goto	STOR_VALUE

; use Jumper flags to determine what setup is required
	
; jumper settings
	movf	JUMPER,w
	xorlw	B'00001110'		; all jumpers JP1, JP2 and JP3 in so START up rate selected
	btfsc	STATUS,Z	
	goto	STO_START		; store startup rate

	movf	JUMPER,w
	xorlw	B'00001100'		; JP2 and JP3 in so FREQ1 selected
	btfsc	STATUS,Z	
	goto	STO_FREQ1		; store startup rate

	movf	JUMPER,w
	xorlw	B'00001010'		;  JP1 and JP3 in so FREQ2 selected
	btfsc	STATUS,Z	
	goto	STO_FREQ2		; store startup rate

	movf	JUMPER,w
	xorlw	B'00000110'		;  JP1  and JP2 in so FREQ3 selected
	btfsc	STATUS,Z	
	goto	STO_FREQ3		; store startup rate
	goto	SET_LOOP		; none

STO_FREQ1
	movf	PROCESSED,w
	movwf	FREQ1			; store in frequency1
	goto	SET_LOOP

STO_FREQ2
	movf	PROCESSED,w
	movwf	FREQ2			; store in frequency2
	goto	SET_LOOP
	
STO_FREQ3
	movf	PROCESSED,w
	movwf	FREQ3			; store in frequency3
	goto	SET_LOOP

STO_START
	movf	ADRESH,w
	movwf	START			; store in start
	rlf		START,f			; use only ls 7-bits
	btfss	STATUS,C		; if carry is set then Doppler	
	goto	CLR_DOPPLER
	movlw	D'03'
	movwf	DOPPLER
	goto	SET_LOOP
CLR_DOPPLER
	clrf		DOPPLER
	goto	SET_LOOP

STOR_VALUE
	call		STORE_VALUES
; re-allow interrupt
	bsf 		INTCON,GIE		; set global interrupt enable 
	goto	SET_LOOP
; _______________________________________________________
STORE_VALUES
; place FREQ1,2,3 in DATA1,2,3 and START in DATA4
	clrf		DATA1ms			; data of first byte ms
	clrf		DATA2ms			; data of second byte ms
	clrf		DATA3ms			; data of third byte ms
	movf	DOPPLER,w
	movwf	DATA4ms			; data of fourth byte ms. for Doppler store
; ls bytes
	movf	FREQ1,w			; frequency 1
	movwf	DATA1ls				; data 1 ls 
 	movf	FREQ2,w			; frequency 2
	movwf	DATA2ls				; data 2 ls 
	movf	FREQ3,w			; frequency 3
	movwf	DATA3ls				; data 3 ls 

	movf	START,w			; startup
	movwf	DATA4ls				; data 1 ls 

; stop interrupt
	bcf 		INTCON,GIE		; clear global interrupt enable 
; write to memory
	call		WRITE
	return
;:______________________________________________________

LOOP_START
; calculate turning points for rectangular wave generation 
WAVE1
	bcf		STATUS,C
	rrf		FREQ1,w
	movwf	DIV1_2			; midpoint of frequency generation
; remark out following 8 lines for genuine square wave
	bcf		STATUS,C
	rrf		DIV1_2,w
	movwf	TEMP1			
	bcf		STATUS,C
	rrf		TEMP1,f
	bcf		STATUS,C
	rrf		TEMP1,w
	subwf	DIV1_2,f
WAVE2
	bcf		STATUS,C
	rrf		FREQ2,w
	movwf	DIV2_2			; midpoint of frequency generation
; remark out following 8 lines for genuine square wave
	bcf		STATUS,C
	rrf		DIV2_2,w
	movwf	TEMP1			
	bcf		STATUS,C
	rrf		TEMP1,f
	bcf		STATUS,C
	rrf		TEMP1,w
	subwf	DIV2_2,f
WAVE3
	bcf		STATUS,C
	rrf		FREQ3,w
	movwf	DIV3_2			; midpoint of frequency generation
; remark out following 8 lines for genuine square wave
	bcf		STATUS,C
	rrf		DIV3_2,w
	movwf	TEMP1			
	bcf		STATUS,C
	rrf		TEMP1,f
	bcf		STATUS,C
	rrf		TEMP1,w
	subwf	DIV3_2,f

LOOP

; check for if started or ready to start
	movf	ON_OFF,w
	btfss	STATUS,Z		; if zero (off) check S1
	goto	RUNNING
; Check S1 
	btfsc	GPIO,3
	goto	LOOP			; no whistle

RUNNING
	clrf		SHIFT_RAMP	; flag used in Doppler Effect for ramping frequency
	bsf		ON_OFF,0		; set for run
; start with normal frequency (changed if Doppler Effect)
	bsf		STATUS,RP0	; bank1
	movlw	B'00000000'
	movwf	OSCTUNE		; normal frequency
	bcf		STATUS,RP0	; bank 0
	
;  Value in START sets whistle start up rate (volume increase) and Doppler start up rate
; read A/D for extended whistle. If <127 then no Doppler, if >127 then Doppler
	call		ACQUIRE_AD	; get VR1 value 
	movf	ADRESH,w
	movwf	LENGTH			; whistle length
	btfss	DOPPLER,0		; if clear then no Doppler effect
	goto	SIREN_LENGTH

;_____________________________________________________________________________________________

DOPPLER_RUN
	clrf		APROCH_AWAY	; flag for direction of volume required. 
	bsf		APROCH_AWAY,0; start at  increasing
	clrf		VOL_RAMP		; position in LNGTH_WKG where volume is to start decreasing
	movf	LENGTH,w		; 
	movwf	LNGTH_WKG	; working length
; set at a minimum of 32
	movf	LNGTH_WKG,w
	sublw	D'32'
	movlw	D'32'
	btfsc	STATUS,C		; if negative keep value, if positive use 32
	movwf	LNGTH_WKG
; divide LNGTH_WKG into approaching, passing and departing (higher, normal and lower frequency shift)
; find centre value-1/16 for shift change from higher to normal
; find centre value +1/16 for change from normal to lower

; divide by 16
	swapf	LNGTH_WKG,w	; swap over and retrieve /16
	andlw	B'00001111'	
	movwf	SXTN			; 1/16 value
; find divide by 2 value
	bcf		STATUS,C
	rrf		LNGTH_WKG,w
	movwf	TEMP
	movwf	MIDWAY			; used in volume

; add SXTN for first frequency shift point and take away SXTN for second frequency shift point
	movf	SXTN,w
	addwf	TEMP,w
	movwf	SHIFT1			; high to normal shift point
; subtract
	movf	SXTN,w
	subwf	TEMP,w
	movwf	SHIFT2			; normal to low shift point
 	
; use higher frequency  initially, reducing to standard 00000, then  slower frequency

; start with high frequency (train coming toward observer)
	movf	SPEED,w		; speed value for higher frequency
	bsf		STATUS,RP0	; bank1
	movwf	OSCTUNE		; maximum frequency
	bcf		STATUS,RP0	; bank 0
	goto	SL1
 ;______________________________________________________________________________________________
	
SIREN_LENGTH ; run whistle sound
	movf	LENGTH,w	 	;  0-255
	btfsc	STATUS,Z		; if zero add 1
	addlw	D'1'
	movwf	LNGTH_WKG	; timing for whistle length
	
SL1	movf	START,w		; start up rate
	movwf	STRT_WKG		; startup working register

; START value sets increase in volume at start
; PWM volume start at D8 and finish at D20, min to max volume from 0.4V to 1V
	movlw	D'8'				; minimum volume start
	movwf	CCPR1L		; ms byte of PWM
	movwf	VOLUME

	bcf		CCP1CON,4
	bcf		CCP1CON,5		; clear ls bits
	bcf		STATUS,C
	rlf		VOLUME,f		; shift left to allow for the two CCP1CON ls bits
	bcf		STATUS,C
	rlf		VOLUME,f

; start  Interrupt for output frequencies generation

	bcf		INTCON,T0IF		;  flag	 
	bsf 		INTCON,GIE		; set global interrupt enable 

SIREN_LOOP
	
; increase volume at start rate
; decrease STRT_WKG (working startup rate) and LNGTH_WKG (working whistle length)
; stop whistle if LNGTH_WKG=0 and S1 open.

	decfsz	EXTENSION,f
	goto	ST_RATE1

	movlw	D'90'				; sets overall range of whistle length with VR1
	call		DELAY2

; check if zero
	movf	LNGTH_WKG,w
	btfsc	STATUS,Z
	goto	CK_S1_1
	decfsz	LNGTH_WKG,f
	goto	DOPPLER_ON_OFF
	goto	CK_S1_1

DOPPLER_ON_OFF

; if Doppler Effect is on check for shift points
	btfss	DOPPLER,0			; if set then check Doppler shifts
	goto	NO_DOPPLER

; check flags for ramping in process
	btfsc	SHIFT_RAMP,1
	goto	ST_RATE1
	btfss	SHIFT_RAMP,0		; if set ramping to 0 (standard calibration frequency) is in process
	goto	CK_SHIFT

TUNE_DOWN
; decrease to zero (normal calibrated frequency) and then lower frequency as train approaches and goes past 
; and change to lower frequency (train departing) toward the value in RAMP_END

	bsf		STATUS,RP0	; bank1
	movf	OSCTUNE,w		; frequency setting
	xorwf	RAMP_END,w	; if the same end the ramping
	btfsc	STATUS,Z
	goto	END_DOWN
; next lower frequency
	decf	OSCTUNE,w
	andlw	B'00011111'		; only 5-bits
	movwf	OSCTUNE

	xorwf	RAMP_END,w	; if the same, end the ramping
	btfsc	STATUS,Z
	goto	END_DOWN
; next lower frequency
	decf	OSCTUNE,w
	movwf	OSCTUNE
	xorwf	RAMP_END,w	; if the same, end the ramping
	btfsc	STATUS,Z
	goto	END_DOWN
	bcf		STATUS,RP0	; bank 0
	goto	ST_RATE1

END_DOWN ; end of further down ramp
	bcf		STATUS,RP0	; bank 0
; when 0 clear SHIFT_RAMP,0 flag
	bcf		SHIFT_RAMP,0 	; ramp up flag off
	bsf		SHIFT_RAMP,1	
	goto	ST_RATE1

CK_SHIFT
	movf	LNGTH_WKG,w
	xorwf	SHIFT1,w
	btfss	STATUS,Z
	goto	ST_RATE1

; set shift flag bit 0 for increase
	bsf		SHIFT_RAMP,0
	comf	SPEED,w		; complementary value for lower frequency
	andlw	B'00011111'
	movwf	RAMP_END		; required frequency setting
	goto	TUNE_DOWN
	
NO_DOPPLER
ST_RATE1; startup rate
; decrease STRT_WKG (down to 0) when EXTENSION,0 is set
	btfss	EXTENSION,0
	goto	SIREN_LOOP

	movf	STRT_WKG,w	; reload at 0
	btfss	STATUS,Z
	goto	S_LOOP		; decrease STRT_WKG
; reload when zero
	movf	START,w		; start up rate
	movwf	STRT_WKG

VOL_ME

; Doppler volume
	btfss	DOPPLER,0		; set for Doppler
	goto	VOLUME_NON_DOPPLER

DOPPLER_VOLUME

; increase to maximum volume or increase untill at half way (MIDWAY) into length. 
; Check where in LNGTH_WKG max volume occurs and use corresponding period to decrease at START rate
; use APROCH_AWAY flag to indicate which way volume is required

	btfss	APROCH_AWAY,0	; when set increase volume
	goto	CHK_DEC_VOL		; check for if decrease in volume is required

; if  LNGTH_WKG reaches MIDWAY  before maximum volume, start reducing volume (set APROCH_AWAY,1)

	movf	LNGTH_WKG,w
	subwf	MIDWAY,w
	btfss	STATUS,C		; if positive, then past midway
	goto	INC_VOL
; and if volume is <20
	movf	CCPR1L,w		; volume
	sublw	D'20'			; when >
	btfss	STATUS,C		; 
	goto	INC_VOL
	bsf		APROCH_AWAY,1
	goto	CHK_DEC_VOL	; check for if decrease in volume is required

INC_VOL
	movf	CCPR1L,w		; volume
	sublw	D'20'			; when >
	btfss	STATUS,C		; was btfss
	goto	CHECK
	incf		VOLUME,f
	btfss	VOLUME,0		; if set set CCP1CON,4
	goto	CLR4
	bsf		CCP1CON,4
	goto	BIT5	
CLR4
	bcf		CCP1CON,4
BIT5
	btfss	VOLUME,1		; if set set CCP1CON,5
	goto	CLR5_V
	bsf		CCP1CON,5
	goto	CCP	
CLR5_V
	bcf		CCP1CON,5
CCP
; volume to CCPR1L
	bcf		STATUS,C
	rrf		VOLUME,w
	movwf	TEMP
	bcf		STATUS,C
	rrf		TEMP,f
	movf	TEMP,w
; increase volume at STRT_WKG (working startup rate) and EXTENSION when processed for every x counts
	sublw	D'20'
	movf	TEMP,w
	btfsc	STATUS,C		; if  20 stop increase
	movwf	CCPR1L		; increase if < 

; if reaches maximum  find position  (LNGTH_WKG). place in VOL_RAMP
; check for 20	
; check
CHECK	
	incf		CCPR1L,w		; volume
	sublw	D'20'			; when > 
	btfsc	STATUS,C
	goto	S_LOOP

; take LNGTH_WKG from LENGTH place in VOL_RAMP
	movf	LNGTH_WKG,w
	subwf	LENGTH,w
	movwf	VOL_RAMP	

; clear APROCH_AWAY
	clrf		APROCH_AWAY	; so volume does not change

CHK_DEC_VOL	; check for if decrease in volume is required
	btfsc	APROCH_AWAY,1	; when set, decrease volume
	goto	DEC_VOL

; compare LNGTH_WKG with VOL_RAMP. If < VOL_RAMP, set APROCH_AWAY,1 to start decreasing volume
	movf	LNGTH_WKG,w
	subwf	VOL_RAMP,w
	btfsc	STATUS,C
	bsf		APROCH_AWAY,1
	goto	S_LOOP
DEC_VOL
; decrease volume to zero
	movf	VOLUME,w
	btfsc	STATUS,Z
	goto	END1
	decfsz	VOLUME,f
	goto	VOL_DOWN
	goto	END1
VOL_DOWN
	btfss	VOLUME,0		; if set, set CCP1CON,4
	goto	CLR4_DEC
	bsf		CCP1CON,4
	goto	BIT5_DEC	
CLR4_DEC
	bcf		CCP1CON,4
BIT5_DEC
	btfss	VOLUME,1		; if set set CCP1CON,5
	goto	CLR5_DEC
	bsf		CCP1CON,5
	goto	CCP_DEC	
CLR5_DEC
	bcf		CCP1CON,5
CCP_DEC
; volume to CCPR1L
	bcf		STATUS,C
	rrf		VOLUME,w
	movwf	TEMP
	bcf		STATUS,C
	rrf		TEMP,f
	movf	TEMP,w
	btfss	STATUS,Z		; if  0 stop decrease
	movwf	CCPR1L		; decrease if < 

	movf	CCPR1L,w		; if zero stop
	btfss	STATUS,Z	
	goto	S_LOOP
END2
	goto	END1
;

VOLUME_NON_DOPPLER
; and if volume is <20

	movf	CCPR1L,w		; volume
	sublw	D'20'			; when >
	btfss	STATUS,C		; 
	goto	SIREN_LOOP

	btfsc	STATUS,Z
	goto	SIREN_LOOP

	incf		VOLUME,f
	btfss	VOLUME,0		; if set set CCP1CON,4
	goto	CLR4_ND
	bsf		CCP1CON,4
	goto	BIT5_ND	
CLR4_ND
	bcf		CCP1CON,4
BIT5_ND
	btfss	VOLUME,1		; if set set CCP1CON,5
	goto	CLR5_V_ND
	bsf		CCP1CON,5
	goto	CCP_ND	
CLR5_V_ND
	bcf		CCP1CON,5
CCP_ND
; volume to CCPR1L
	bcf		STATUS,C
	rrf		VOLUME,w
	movwf	TEMP
	bcf		STATUS,C
	rrf		TEMP,f
	movf	TEMP,w
; increase volume at STRT_WKG (working startup rate) and EXTENSION when processed for every x counts
	sublw	D'20'
	movf	TEMP,w
	btfsc	STATUS,C		; if  20 stop increase
	movwf	CCPR1L		; increase if < 
	goto	SIREN_LOOP

S_LOOP
	decf	STRT_WKG,f
	goto	SIREN_LOOP

CK_S1_1
; if S1 closed continue
	btfss	GPIO,3;SW_READ,3		; switch read in interrupt
	goto	ST_RATE1	

; S1 open so stop

STOP1
	movlw	D'5'				; whistle off rate
	call		DELAY2
; decrease volume to zero
	movf	CCPR1L,w		; if zero stop
	btfsc	STATUS,Z	
	goto	END1			;  end of whistle sound
	decf	CCPR1L,f		; decrease volume
	goto	STOP1			; decrease until zero
END1
	bcf 		INTCON,GIE		; clear global interrupt enable 
	clrf		COUNT1		; counters at 0
	clrf		COUNT2
	clrf		COUNT3
	clrf		ON_OFF		; whistle off
	clrf		EXTENSION		; length extension
	movlw	D'255'
	call		DELAY2
	goto	LOOP

; ****************************************************
; subroutines

; subroutine to wait for A/D conversion
ACQUIRE_AD
	bsf		ADCON0,1	; GO/DONE bit start conversion
WAIT_CONV
	btfsc	ADCON0,1	; conversion complete when cleared ~11 cycles
	goto	WAIT_CONV
	return

DELAY1
	movlw	D'125'	; 100ms
DELAY2
DELAYX
	movwf	STORE1
LOOP2
	movlw	D'255'
	movwf	STORE2
LOOP1
	decfsz	STORE2,f
	goto	LOOP1
	decfsz	STORE1,f
	goto	LOOP2
	return

; read data memory
READ
	bsf		STATUS,RP0	; select memory bank 1 
	movlw 	H'04'
	movwf 	PMADRH		; ms Byte of Program Address to read
	movf	TEMP,w
	movwf 	PMADRL 		; ls Byte of Program Address to read

	bsf	 	PMCON1,RD 	; Read
	nop 					
	nop
; memory is read in second cycle PM read instruction
	movf	PMDATH,w 		; ms Byte of Program data 
	movwf	TEMP
; only require ls byte for this particular code
	movf	PMDATL,w 		; ls Byte of Program data
	bcf		STATUS,RP0	; bank 0
	return

; write to data memory
WRITE
; This write routine assumes the following:
;
	bsf		STATUS,RP0	; select memory bank 1
	movlw 	H'04'
	movwf 	PMADRH		; MS Byte of Program Address to write
	movlw	H'00'
	movwf 	PMADRL 		; LS Byte of Program Address to write

	movlw	H'78'			; Load initial data address
	movwf	FSR 
LOOP_WRITE 
	movf	 INDF,w 			; Load first data byte into upper byte
	movwf	 PMDATH		;
	incf		 FSR,f 			; Next byte
	movf	 INDF,w 			; Load second data byte into lower byte
	movwf	 PMDATL 		; 		
	incf		 FSR,f 			;
	bsf		 PMCON1,WREN ; Enable writes

CHECK_GIE; comment out if not using interrupt whilst writing
	bcf		 INTCON,GIE 	; Disable interrupts 
	btfsc	 INTCON,GIE 	; check if disabled
	goto	CHECK_GIE
;
; Required Sequence
	movlw	H'55'			; Start of required write sequence:
	movwf	PMCON2		; Write 55h
	movlw	H'AA'		 	;
	movwf	PMCON2 		; Write AAh
	bsf		PMCON1,WR 	; Set WR bit to begin write
	nop					 	; NOPs required for time to transfer data to the buffer registers
	nop						; 
;
	bcf		 PMCON1,WREN ; Disable writes
;	bsf		 INTCON,GIE 	; Re-enable interrupts (comment out if not using interrupts)

	movf	PMADRL,w
	incf		PMADRL,f		; Increment address
; select 4,8,12 or16 	
	; 0F = 16 words
	; 0B = 12 words
	; 07 =  8 words
	; 03 =  4 words
	andlw	H'03'			; Indicates when the set number of words have been programmed
	sublw	H'03'

	btfss	STATUS,Z 		; Exit on a match,
	goto	LOOP_WRITE	; Continue until ended
	bcf		STATUS,RP0	; bank 0
	return

; *************************************************
	end


